/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <system/Processes.h>

#include <model/QuteCom.h>
#include <model/config/CommandLineParser.h>
#include <model/config/Config.h>
#include <model/config/ConfigImporter.h>
#include <model/config/ConfigManagerFileStorage.h>
#include <model/config/ConfigManager.h>
#include <model/config/StartupSettingListener.h>
#include <model/classic/ClassicExterminator.h>
#include <model/commandserver/CommandClient.h>
#include <locale>

#include <control/CQuteCom.h>

#ifdef GTKINTERFACE
	#include <presentation/gtk/GtkFactory.h>
#else
	#include <presentation/qt/QtFactory.h>
	#include <presentation/qt/QtLanguage.h>
#endif

#include <imwrapper/IMWrapperFactory.h>
#include <sipwrapper/SipWrapperFactory.h>
#include <sipwrapper/SipWrapper.h>

#include <QuteComBuildId.h>

#ifdef PHAPIWRAPPER
	#include <PhApiFactory.h>
	#include <../src/null/NullIMFactory.h>
#elif defined(SIPXWRAPPER)
	#include <SipXFactory.h>
	#include <NullIMFactory.h>
#elif defined (MULTIIMWRAPPER)
	#include <PhApiFactory.h>
	#include <multiim/MultiIMFactory.h>
	#include <PurpleIMFactory.h>
#else
	#include <NullSipFactory.h>
	#include <NullIMFactory.h>
#endif

#include <util/File.h>
#include <util/Logger.h>
#include <util/Path.h>
#include <util/SafeDelete.h>
#include <util/WebBrowser.h>

#include <system/RegisterProtocol.h>

#include <qtutil/FSResourceFileEngineHandler.h>

#include <cutil/global.h>

#if defined(ENABLE_CRASHREPORT)
#if defined(CC_MSVC)
	#include <memorydump/MSVCMemoryDump.h>
#elif defined(CC_MINGW)
	#include <winsock2.h>
#elif defined(OS_LINUX)
	#include <memorydump/UNIXMemoryDump.h>
#elif defined(OS_MACOSX)
	#include <memorydump/MACMemoryDump.h>
#endif
#endif // ENABLE_CRASHREPORT

#if defined(OS_MACOSX) || defined(OS_LINUX)
	#include <signal.h>
	#include <sys/wait.h>
#endif

#include <QtGui/QtGui>

#include <sstream>

#include <buildconfig.h>

#if defined(OS_LINUX)
    #include <X11/Xlib.h>
#endif

static const char* CONFIG_SUBDIR = "/config/";


/**
 * Helper class to instantiate and delete factories based on the compilation
 * flags.
 */
struct FactoryHelper {
	FactoryHelper() {
	#ifdef SIPXWRAPPER
		_sipFactory = new SipXFactory();
		_imFactory = new NullIMFactory();

		SipWrapperFactory::setFactory(_sipFactory);
		IMWrapperFactory::setFactory(_imFactory);

	#elif defined(PHAPIWRAPPER)
		PhApiFactory* phApiFactory = new PhApiFactory();
		_sipFactory = phApiFactory;
		
		_imFactory = phApiFactory;
		
		SipWrapperFactory::setFactory(phApiFactory);
		IMWrapperFactory::setFactory(_imFactory);

	#elif defined(MULTIIMWRAPPER)
		PhApiFactory* phApiFactory = new PhApiFactory();
		_sipFactory = phApiFactory;
		_purpleIMFactory = new PurpleIMFactory();
		_imFactory = new MultiIMFactory(*phApiFactory, *_purpleIMFactory);
		
		SipWrapperFactory::setFactory(phApiFactory);
		IMWrapperFactory::setFactory(_imFactory);
	

	#else
		_sipFactory = new NullSipFactory();
		_imFactory = new NullIMFactory();

		SipWrapperFactory::setFactory(_sipFactory);
		IMWrapperFactory::setFactory(_imFactory);

		
	#endif

	}

	~FactoryHelper() {
		OWSAFE_DELETE(_sipFactory);
#ifndef PHAPIWRAPPER
		OWSAFE_DELETE(_imFactory);
#endif
		
	#ifdef MULTIIMWRAPPER
		OWSAFE_DELETE(_purpleIMFactory);
	#endif
	}

	IMWrapperFactory * _imFactory;
	SipWrapperFactory * _sipFactory;

#if defined(MULTIIMWRAPPER)
	PurpleIMFactory* _purpleIMFactory;
#endif

};


/**
 * Stub function to make GCC silent.
 *
 * @see http://www-eleves-isia.cma.fr/documentation/BoostDoc/boost_1_29_0/libs/test/doc/minimal.htm
 */
int test_main(int argc, char *argv[]) {
	return 1;
}

static void initLogger(const std::string& userConfigDir) {
	std::string logFileName = userConfigDir + BINARY_NAME + ".log";
	// create directory if it doesn't exist
	if (!File::exists(userConfigDir)) {
		File::createPath(userConfigDir);
	}
	if (!File::isDirectory(userConfigDir)) {
		LOG_FATAL("User configuration dir '" + userConfigDir + "' does not exist or is not a directory");
	}

	Logger::getInstance()->setLogFileName(logFileName);
}

static void initConfig(const CommandLineParser& cmdLineParser) {
	ConfigManagerFileStorage configManagerStorage(ConfigManager::getInstance());

	std::string resourcesDir = cmdLineParser.getResourcesDir();
	if (!File::isDirectory(resourcesDir)) {
		LOG_FATAL("Resources dir '" + resourcesDir + "' does not exist or is not a directory");
	}

	std::string userConfigDir = cmdLineParser.getUserConfigDir();
	configManagerStorage.loadSystemConfig(resourcesDir + CONFIG_SUBDIR);
	Config::setConfigDir(userConfigDir);
	ConfigImporter importer;
	importer.importConfig();
	configManagerStorage.loadUserConfig(userConfigDir);

	Config & config = ConfigManager::getInstance().getCurrentConfig();
	config.setResourcesDir(resourcesDir);
	WebBrowser::setBrowser(config.getLinuxPreferedBrowser());
}

static void registerHyperlinkProtocol(const Config& config, const std::string& executableName) {
	RegisterProtocol registerProtocol(config.getHyperlinkProtocol());
	std::string executableFullName = Path::getApplicationDirPath() + executableName;
	registerProtocol.bind(executableFullName + " -c %1", executableFullName + ",0", config.getCompanyWebSiteUrl());
}

/**
 * Callback additional info for MemoryDump.
 */
std::string getAdditionalInfo() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	std::stringstream stream;
	stream
		<< "User: " << config.getProfileLastUsedName() << std::endl
		<< "buildid: " << QuteComBuildId::getBuildId() << std::endl
		<< "revision: " << QuteComBuildId::getSvnRevision() << std::endl;
	return stream.str();
}

#if defined(OS_MACOSX) || defined(OS_LINUX)
static void sigpipe_catcher(int sig) {
#ifndef NDEBUG
	// Do not use LOG_DEBUG. There is only a limited set of functions you are
	// allowed to call from withing a signal catcher. See signal man page.
	static char msg[] = "Caught signal: SIGPIPE\n";
	write(2, msg, sizeof(msg));
#endif
}
#endif

int main(int argc, char * argv[]) {
	//Init presentation factories before parsing the command line so that Qt or
	//Gtk get a chance to parse their command line options ('-style' for Qt for
	//example)
        int nocrashreport = 0;
#if defined(OS_MACOSX)
        std::locale  lc;
#else
        std::locale lc("");
#endif
        std::locale::global(lc);

	PFactory * pFactory = NULL;
#if defined(OS_LINUX)
	XInitThreads();
#endif

#ifdef GTKINTERFACE
	pFactory = new GtkFactory(argc, argv);
#else
	pFactory = new QtFactory(argc, argv);
#endif
        
#if defined(OS_MACOSX)
        QDir dir(QApplication::applicationDirPath());
        dir.cdUp();
        dir.cd("plugins");
        QApplication::setLibraryPaths(QStringList(dir.absolutePath()));
#endif
        
	PFactory::setFactory(pFactory);

	CommandLineParser cmdLineParser(BINARY_NAME, argc, argv);
	initLogger(cmdLineParser.getUserConfigDir());
	LOG_DEBUG("Started");

#if defined(OS_MACOSX) || defined(OS_LINUX)
	signal(SIGPIPE, sigpipe_catcher);
#endif

	// Setup factories. These must be initialized before loading config because
	// it's used when importing configuration from the Classic version.
	FactoryHelper factoryHelper;

	initConfig(cmdLineParser);

	Config & config = ConfigManager::getInstance().getCurrentConfig();

	SipWrapper* w = factoryHelper._sipFactory->createSipWrapper();
	
	w->setSipOptions("sip.register_timeout", std::string(QString::number(config.getSipRegisterTimeOut()).toUtf8()));
	w->setSipOptions("sip.publish_timeout", std::string(QString::number(config.getSipPublishTimeOut()).toUtf8()));
	if(config.getSipUseOptions())
		w->setSipOptions("sip.use_options_request", "true");
	else
		w->setSipOptions("sip.use_options_request", "false");

	if(config.getSipP2pPresence())
		w->setSipOptions("sip.p2p_presence", "true");
	else
		w->setSipOptions("sip.p2p_presence", "false");

	if(config.getSipChatWithoutPresence())
		w->setSipOptions("sip.chat.without.presence", "true");
	else
		w->setSipOptions("sip.chat.without.presence", "false");

	if(config.getSipUseTypingState())
		w->setSipOptions("sip.use_typing_state", "true");
	else
		w->setSipOptions("sip.use_typing_state", "false");


	//Remove QuteCom Classic from startup registry
	ClassicExterminator::killClassicExecutable();

	//Remove QuteCom Classic from startup registry
	ClassicExterminator::removeClassicFromStartup();

#ifdef ENABLE_CRASHREPORT
	for(int i = 1; i < argc; i++) {
	  if (!strcmp(argv[i], "-nocrash")) {
	    nocrashreport = 1;
	    break;
	  }
	}

	if (!nocrashreport) {
    #if defined(CC_MSVC)
	  MSVCMemoryDump * memoryDump = new MSVCMemoryDump("QuteCom", QuteComBuildId::getSvnRevision());
	// This is needed for CRT to not show dialog for invalid param
	// failures and instead let the code handle it.
	  _CrtSetReportMode(_CRT_ASSERT, 0);
    #endif

    #if defined(OS_MACOSX)
	  MACMemoryDump * memoryDump = new MACMemoryDump("QuteCom", QuteComBuildId::getSvnRevision());
    #endif

    #if defined(OS_LINUX)
	  UNIXMemoryDump * memoryDump = new UNIXMemoryDump("QuteCom", QuteComBuildId::getSvnRevision());
    #endif

	  memoryDump->setGetAdditionalInfo(getAdditionalInfo);
	}

#endif // ENABLE_CRASHREPORT

	// Uncomment to test crash report
	
	/*int* pointer;
	pointer = 0;
	*pointer = 12;*/
	

	//No 2 qtqutecom at the same time
	if (Processes::isRunning(BINARY_NAME) && !cmdLineParser.isSeveralQuteComAllowed()) {
		const std::string callFromCommandLine = cmdLineParser.getCommand();
		CommandClient client;
		client.connect();
		if (!callFromCommandLine.empty()) {
			client.call(callFromCommandLine);
		} else {
			client.bringToFront();
		}
		return EXIT_SUCCESS;
	}

	// Register protocol used to associate the application in HTML links
	registerHyperlinkProtocol(config, cmdLineParser.getExecutableName());

	// Make the application starts when user logs on computer
	StartupSettingListener settingListener(cmdLineParser.getExecutableName());

	// Init file system resource engine
	FSResourceFileEngineHandler handler(QString::fromUtf8(config.getResourcesDir().c_str()));

	// CQuteCom creates PQuteCom (QtQuteCom, GtkQuteCom...)
	// and then starts the model thread. This way the gui is shown as soon as
	// possible
	QuteCom & qutecomPhone = QuteCom::getInstance();
	qutecomPhone.setStartupCall(cmdLineParser.getCommand());
	CQuteCom cQuteCom(qutecomPhone, cmdLineParser.getRunInBackground()||config.getGeneralStartBackgroundMode());
	pFactory->exec();

	QuteCom::deleteInstance();
	LOG_DEBUG("Ended");

	return EXIT_SUCCESS;
}
